// server/server.js
// KawduLive — Simple WebRTC Signaling + Matchmaking Server

const express = require('express');
const http = require('http');
const cors = require('cors');
const { Server } = require('socket.io');

const app = express();
app.use(cors());
app.use(express.json());

const server = http.createServer(app);
const io = new Server(server, {
  cors: { origin: '*' }
});

const PORT = process.env.PORT || 4000;

// In-memory waiting queue (replace with Redis in production)
let waiting = null;

app.get('/status', (req, res) => {
  res.json({
    app: "KawduLive Signaling Server",
    clients: io.engine.clientsCount,
    waiting: !!waiting
  });
});

// Moderation endpoint (placeholder)
app.post('/report', (req, res) => {
  console.log('KawduLive report received:', req.body);
  res.json({ ok: true });
});

io.on('connection', socket => {
  console.log('KawduLive client connected:', socket.id);

  socket.on('find_match', () => {
    console.log('KawduLive find_match:', socket.id);

    if (waiting && waiting !== socket.id) {
      const peerId = waiting;
      waiting = null;

      const room = `${peerId}#${socket.id}`;
      socket.join(room);
      io.to(peerId).socketsJoin(room);

      io.to(peerId).emit('matched', { room, initiator: false, peer: socket.id });
      socket.emit('matched', { room, initiator: true, peer: peerId });

      console.log('KawduLive paired:', peerId, socket.id, 'room:', room);
    } else {
      waiting = socket.id;
      socket.emit('waiting');
      console.log('KawduLive waiting queue:', waiting);
    }
  });

  socket.on('signal', ({ room, data }) => {
    socket.to(room).emit('signal', { from: socket.id, data });
  });

  socket.on('leave', ({ room }) => {
    socket.leave(room);
    socket.to(room).emit('peer_left');
  });

  socket.on('disconnect', () => {
    if (waiting === socket.id) waiting = null;
    console.log('KawduLive disconnect:', socket.id);
  });
});

server.listen(PORT, () => {
  console.log(`KawduLive signaling server running on http://localhost:${PORT}`);
});
